(function($) {

// The random choice urns into which the banners will be loaded
var urns = {};
var urn_ids = [];

/*
 * This function puts a campaign in a specific urn.
 *
 * @param urn the random urn to put the campaign in.
 * @param campaign a valid campaign to run. A campaign object must have an
 *                 image field, title field and link field.
 */
function put(urn, campaign) {
    if(!(urn in urns)) {
        urns[urn] = [];
        urn_ids.push(urn);
    }
    urns[urn].push(campaign);
}

// === PRELOADED CAMPAIGNS ===

// UCN campaign, approved for magic link
put('ic:union', {
    image: 'user:ucn.png',
    title: 'Union Central News: Faster, better and on a holoscreen near you!',
    link: '/phpbb/viewtopic.php?f=5&t=100&start=999999#lastPost',
    approved: true
});

// === Internals ===
//

// Automate the process of banner handling:
$(function() {
    banner_load(function() {
        var picked = banner_pick();
        banner_update(picked);
    });
});

/*
 * Loads banners into urns.
 */
function banner_load(callback) {
    function getval(obj, key, def) {
        if(typeof(def) == 'undefined') {
            def = null;
        }
        var prop = key.toLowerCase();
        for(var p in obj) {
            if(obj.hasOwnProperty(p) && prop == p.toLowerCase()) {
                if(obj[p].length > 0) {
                    return obj[p][0];
                }
            }
        }
        return def;
    }

    $.get('/api/data/entities/banner_campaign')
    .then(function(data) {
        var result = {};

        var now = (new Date())/1000;
        $.each(data, function(id, camp) {
            // no banner
            if(!getval(camp, 'banner')) return;
            // no title or link
            if(!getval(camp, 'link') || !getval(camp, 'title')) return;

            var start = getval(camp, 'start');
            if(!!start) start = parseInt(start);
            var end = getval(camp, 'end');
            if(!!end) end = parseInt(end);

            // has start, start is in past, no end, or end is in future
            if(!!start && now > start && (!end || now < end)) {
                result[id] = camp;
            }
        });

        loaded('campaigns', result);
    });

    $.get('/api/data/entities/corporation')
    .then(function(data) {
        var result = {};

        $.each(data, function(id, corp) {
            if(!!getval(corp, 'banner')) {
                result[id] = corp;
            }
        });

        loaded('corporations', result);
    });

    var cache = {};

    function loaded(type, data) {
        cache[type] = data;

        // barrier requiring both parts loaded
        if(!('campaigns' in cache) || !('corporations' in cache)) {
            return;
        }

        $.each(cache.campaigns, function(_, c) {
            var user = getval(c, 'user');
            var campaign = {
                image: getval(c, 'banner'),
                title: getval(c, 'title'),
                link: getval(c, 'link')
            };
            put(user, campaign);
        });

        $.each(cache.corporations, function(id, c) {
            var user = getval(c, 'user');
            var campaign = {
                image: getval(c, 'banner'),
                title: getval(c, 'entry title'),
                link: id
            };
            put(user, campaign);
        });

        callback();
    }
}

/*
 * Picks a campaign. Chances of picking an urn are uniformly distributed. Then
 * picks a campaign from the selected urn.
 */
function banner_pick() {
    var urn = urns[urn_ids[Math.floor(Math.random()* urn_ids.length)]];
    var campaign = urn[Math.floor(Math.random()* urn.length)];

    return campaign;
}

/*
 * Updates the UI to reflect the given campaign.
 *
 * @param campaign a campaign object.
 */
function banner_update(campaign) {

    function clean_id(id) {
        id = id.toLowerCase();
        id = id.replace(/^\s+/,'').replace(/\s+$/,'');
        id = id.replace(/[^-_.:a-zA-Z0-9]/g,'_');
        id = id.replace(/_+/g,'_');
        id = id.replace(/:+/g,':');
        id = id.replace(/^[-:._]+/,'').replace(/[-:._]+$/,'');
        id = id.replace(/:[-:._]+/g,':');
        id = id.replace(/\s+/g,'_');
        return id;
    }

    // get elements
    var el = document.getElementById('fwurgbanner');
    var al = document.getElementById('fwurgbannerlink');
    
    // sanity check: do we have the requisite elements?
    if(el && al) {
        // set image
        el.style.backgroundImage = 'url(' + '/dokuwiki/_media/' + clean_id(campaign.image) + ')';
        el.className = 'banner';

        // set title
        el.setAttribute('title', campaign.title);

        // set the link target
        var link = '#';

        // first check if this one is flagged as approved (if so we trust the
        // campaign's link implicitly), then check if it should be interpreted
        // as a forum id
        if('approved' in campaign && campaign.approved) {
            // approved campaigns are fine, otherwise use link massaging
            link = campaign.link;
        } else if(!isNaN(parseInt(campaign.link))) {
            // it seems to be a forum id, so use it as such
            link = '/phpbb/viewtopic.php?t=' + parseInt(campaign.link);
        } else {
            // it is something else, clean it and use as wiki id
            link = '/dokuwiki/' + clean_id(campaign.link);
        }

        al.setAttribute('href', link);
    }
}

})(jQuery);
